//=============================================================================
// MPP_MaskFog_Op1.js
//=============================================================================
// Copyright (c) 2022 - 2024 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc Regions change the transparency of that tile.
 * @author Mokusei Penguin
 * @url 
 * 
 * @base MPP_MaskFog
 * @orderAfter MPP_MaskFog
 *
 * @help [version 1.1.0]
 * This plugin is for RPG Maker MZ.
 * 
 * ▼ Plugin parameter details
 *  〇 About transparency level
 *   - You can set the transparency of that tile with the region ID.
 *   - The higher the level, the higher the transparency.
 *   
 *  〇 Region ID array specification
 *   - When setting numbers in an array, you can specify numbers from n to m
 *     by writing n-m.
 *         Example: 1-4,8,10-12
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠ is half-width)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @param Blur
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 999
 *      @default 24
 *
 *  @param Transparent Level 1 Regions
 *      @desc 
 *      @default 1,9,17,25,33,41,49,57
 *
 *  @param Transparent Level 2 Regions
 *      @desc 
 *      @default 2,10,18,26,34,42,50,58
 *
 *  @param Transparent Level 3 Regions
 *      @desc 
 *      @default 3,11,19,27,35,43,51,59
 *
 *  @param Transparent Level 4 Regions
 *      @desc 
 *      @default 4,12,20,28,36,44,52,60
 *
 *  @param Transparent Level 5 Regions
 *      @desc 
 *      @default 5,13,21,29,37,45,53,61
 *
 *  @param Transparent Level 6 Regions
 *      @desc 
 *      @default 6,14,22,30,38,46,54,62
 *
 *  @param Transparent Level 7 Regions
 *      @desc 
 *      @default 7,15,23,31,39,47,55,63
 *
 */

/*:ja
 * @target MZ
 * @plugindesc リージョンでそのタイルの透過率が変わります。
 * @author 木星ペンギン
 * @url 
 * 
 * @base MPP_MaskFog
 * @orderAfter MPP_MaskFog
 *
 * @help [version 1.1.0]
 * このプラグインはRPGツクールMZ用です。
 * 
 * ▼ プラグインパラメータ 詳細
 *  〇 透過レベルについて
 *   - リージョンIDでそのタイルの透過率を設定できます。
 *   - レベルが高いほど透過率が高くなります。
 *   
 *  〇 リージョンIDの配列指定
 *   - 数値を配列で設定する際、n-m と表記することでnからmまでの数値を指定できます。
 *       例 : 1-4,8,10-12
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠は半角)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @param Blur
 *      @text ぼかし
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 999
 *      @default 24
 *
 *  @param Transparent Level 1 Regions
 *      @text 透過レベル1リージョン
 *      @desc 
 *      @default 1,9,17,25,33,41,49,57
 *
 *  @param Transparent Level 2 Regions
 *      @text 透過レベル2リージョン
 *      @desc 
 *      @default 2,10,18,26,34,42,50,58
 *
 *  @param Transparent Level 3 Regions
 *      @text 透過レベル3リージョン
 *      @desc 
 *      @default 3,11,19,27,35,43,51,59
 *
 *  @param Transparent Level 4 Regions
 *      @text 透過レベル4リージョン
 *      @desc 
 *      @default 4,12,20,28,36,44,52,60
 *
 *  @param Transparent Level 5 Regions
 *      @text 透過レベル5リージョン
 *      @desc 
 *      @default 5,13,21,29,37,45,53,61
 *
 *  @param Transparent Level 6 Regions
 *      @text 透過レベル6リージョン
 *      @desc 
 *      @default 6,14,22,30,38,46,54,62
 *
 *  @param Transparent Level 7 Regions
 *      @text 透過レベル7リージョン
 *      @desc 
 *      @default 7,15,23,31,39,47,55,63
 *
 */

(() => {
    'use strict';
    
    const pluginName = 'MPP_MaskFog_Op1';

    /**
     * start から end までの整数を返すイテレータ。
     * 
     * @param {number} start - 最初の数値。
     * @param {number} end - 最後の数値（範囲には含まれない）。
     * @returns {number}
     */
    const range = function* (start, end) {
        for (let i = start; i < end; i++) {
            yield i;
        }
    };

    // Plugin Parameters
    const parameters = PluginManager.parameters(pluginName);
    const convertToSet = (param) => {
        return param.split(',').reduce((r, item) => {
            if (item) {
                const match = /(\d+)-(\d+)/.exec(item);
                if (match) {
                    for (const n of range(+match[1], +match[2] + 1)) {
                        r.add(n);
                    }
                } else {
                    r.add(+item);
                }
            }
            return r;
        }, new Set());
    };
    const param_Blur = Number(parameters['Blur'] || 24);
    const param_TransparentRegions = [null];
    for (let i = 1; i < 8; i++) {
        const param = parameters[`Transparent Level ${i} Regions`];
        param_TransparentRegions[i] = convertToSet(param);
    }

    //-------------------------------------------------------------------------
    // Tilemap

    const _Tilemap_createFogMaskSprite = Tilemap.prototype.createFogMaskSprite;
    Tilemap.prototype.createFogMaskSprite = function() {
        _Tilemap_createFogMaskSprite.call(this);
        this._fogMaskSprite.setBackBlur(param_Blur);
    };

    const _Tilemap_updateTransform = Tilemap.prototype.updateTransform;
    Tilemap.prototype.updateTransform = function() {
        if (this._fogMaskSprite) {
            const ox = Math.ceil(this.origin.x);
            const oy = Math.ceil(this.origin.y);
            const startX = Math.floor((ox - this._margin) / this.tileWidth);
            const startY = Math.floor((oy - this._margin) / this.tileHeight);
            const mx = startX * this.tileWidth - ox;
            const my = startY * this.tileHeight - oy;
            this._fogMaskSprite.setOrigin(mx, my);
            if (
                this._needsRepaint ||
                this._lastStartX !== startX ||
                this._lastStartY !== startY
            ) {
                this._paintFogLayer(startX, startY);
            }
        }
        _Tilemap_updateTransform.apply(this, arguments);
    };

    Tilemap.prototype._paintFogLayer = function(startX, startY) {
        this._fogMaskSprite.clearBack();
        const widthWithMargin = this.width + this._margin * 2;
        const heightWithMargin = this.height + this._margin * 2;
        const tileCols = Math.ceil(widthWithMargin / this.tileWidth) + 1;
        const tileRows = Math.ceil(heightWithMargin / this.tileHeight) + 1;
        for (let y = 0; y < tileRows; y++) {
            for (let x = 0; x < tileCols; x++) {
                this._paintFogSpot(startX, startY, x, y);
            }
        }
    };

    Tilemap.prototype._paintFogSpot = function(startX, startY, x, y) {
        const mx = startX + x;
        const my = startY + y;
        const level = this._fogTransparentLevel(mx, my);
        if (level > 0) {
            const w = this.tileWidth;
            const h = this.tileHeight;
            this._fogMaskSprite.paintSpot(x * w, y * h, w, h, level / 7);
        }
    };

    Tilemap.prototype._fogTransparentLevel = function(x, y) {
        const regionId = this._readMapData(x, y, 5);
        const level = param_TransparentRegions.findIndex(set => set && set.has(regionId));
        return Math.max(level, 0);
    };

})();
